<?php

namespace Elementor;

defined('ABSPATH') or exit;

/**
 * Widget Name: Logo
 */
class Chakavak_Logo extends Widget_Base
{
    const NAME = 'chk_logo';
    const CATEGORY = 'chakavak_header_widgets';

    public function get_name(): string
    {
        return self::NAME;
    }

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_title()
    {
        return __('Logo', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-logo';
    }

    public function get_keywords()
    {
        return ['logo', esc_html__('Logo', 'chakavak')];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'content_section',
            [
                'label' => __('Logo', 'chakavak'),
            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label' => __('Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => chakavak_alignment_2(),
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'logo_type',
            [
                'label' => esc_html__('Logo type', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'image' => [
                        'title' => esc_html__('Image', 'chakavak'),
                        'icon' => 'eicon-image',
                    ],
                    'text' => [
                        'title' => esc_html__('Text', 'chakavak'),
                        'icon' => 'eicon-font',
                    ]
                ],
                'default' => 'image',
            ]
        );

        $this->add_control(
            'logo_image',
            [
                'label' => esc_html__('Image', 'chakavak'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => get_template_directory_uri() . '/assets/images/logo.png',
                ],
                'condition' => [
                    'logo_type' => 'image'
                ],
            ]
        );

        $this->add_responsive_control(
            'logo_width',
            [
                'label' => __('Width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 400,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 1,
                        'max' => 100,
                        'step' => 0.1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-header__logo-image img:first-child' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'logo_type' => 'image'
                ],
            ]
        );

        $this->add_control(
            'logo_sticky_heading',
            [
                'label' => esc_html__('Logo sticky', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'logo_sticky_image',
            [
                'label' => esc_html__('Image', 'chakavak'),
                'type' => Controls_Manager::MEDIA,
                'condition' => [
                    'logo_type' => 'image'
                ],
            ]
        );

        $this->add_responsive_control(
            'logo_sticky_width',
            [
                'label' => __('Width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 400,
                        'step' => 1,
                    ],
                    'rem' => [
                        'min' => 1,
                        'max' => 100,
                        'step' => 0.1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-header__logo-image img.chk-logo__sticky' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'logo_type' => 'image',
                    'logo_sticky_image[url]!' => '',
                ],
            ]
        );

        $this->add_control(
            'logo_text',
            [
                'label' => esc_html__('Name', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Chakavak', 'chakavak'),
                'condition' => [
                    'logo_type' => 'text',
                ],
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );
        $this->add_control(
            'logo_text_tag',
            [
                'label' => __('HTML tag', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => chakavak_heading(),
                'default' => 'h4',
                'condition' => [
                    'logo_type' => 'text',
                ],
            ]
        );

        $this->end_controls_section();

        //Text Logo
        $this->start_controls_section(
            'logo_text_style',
            [
                'label' => esc_html__('Logo text style', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
                    'logo_type' => 'text',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'logo_typography',
                'selector' => '{{WRAPPER}} .chk-header__logo-text',
            ]
        );

        $this->add_control(
            'logo_color',
            [
                'label' => esc_html__('Text color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-header__logo-text' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'logo_color_hover',
            [
                'label' => esc_html__('Text color hover', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-header__logo-text:is(:hover,:focus)' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div class="chk-header__logo lh-1 chk-header-item">
                <?php
                if ($settings['logo_type'] == 'text') {
                    $logo_tag = $settings['logo_text_tag'];
                    $logo_class = 'chk-header__logo-text d-inline-block';
                    $this->add_render_attribute('logo_text', 'class', $logo_class);
                    $this->add_inline_editing_attributes('logo_text', 'none');

                    printf(
                        '<%1$s><a %2$s href="%3$s">%4$s</a></%1$s>',
                        $logo_tag,
                        $this->get_render_attribute_string('logo_text'),
                        esc_url(home_url('/')),
                        esc_html($settings['logo_text'])
                    );
                } elseif ($settings['logo_type'] == 'image') {
                    $static_logo_class = $settings['logo_sticky_image']['url'] ? 'max-w-full h-auto chk-header__logo-static' : '';

                    echo '<a href="' . esc_url(home_url('/')) . '" class="chk-header__logo-image d-inline-block lh-0">';
                    printf(
                        '<img %s src="%s" alt="%s">',
                        $static_logo_class ? 'class="' . $static_logo_class . '"' : '',
                        $settings['logo_image']['url'],
                        esc_attr(get_bloginfo('name', 'display'))
                    );

                    if ($settings['logo_sticky_image']['url']) {
                        printf(
                            '<img class="chk-header__logo-sticky max-w-full h-auto" src="%s" alt="%s">',
                            $settings['logo_sticky_image']['url'],
                            esc_attr(get_bloginfo('name', 'display'))
                        );
                    }
                    echo '</a>';
                }
                ?>
            </div>
        </div>
<?php
    }
}

// After the chakavak_Logo class is defined, I must register the new widget class with Elementor:
Plugin::instance()->widgets_manager->register(new Chakavak_Logo());
