<?php

namespace Elementor;

defined('ABSPATH') or exit;

/**
 * Widget Name: Contact form 7
 */
class chakavak_CF7 extends Widget_Base
{
    const NAME = 'chk_cf7';
    const CATEGORY = 'chakavak_widgets';

    public function get_name()
    {
        return self::NAME;
    }

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_title()
    {
        return __('Contact Form 7', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-form-horizontal';
    }

    public function get_keywords()
    {
        return ['contact', 'contact form', 'cf7', 'form', __('contact form', 'chakavak')];
    }

    public function get_script_depends()
    {
        return ['chk-widgets'];
    }

    protected function get_cf7_forms()
    {

        $field_options = array();

        if (class_exists('WPCF7_ContactForm')) {
            $args = array(
                'post_type' => 'wpcf7_contact_form',
                'posts_per_page' => -1,
            );
            $forms = get_posts($args);
            $field_options['0'] = 'Select';
            if ($forms) {
                foreach ($forms as $form) {
                    $field_options[$form->ID] = $form->post_title;
                }
            }
        }

        if (empty($field_options)) {
            $field_options = array(
                '-1' => esc_html__('Contact Form not found. Please add contact form from Dashboard > Contact > Add New.', 'chakavak'),
            );
        }

        return $field_options;
    }

    protected function get_cf7_form_id()
    {
        if (class_exists('WPCF7_ContactForm')) {
            $args = array(
                'post_type' => 'wpcf7_contact_form',
                'posts_per_page' => -1,
            );
            $forms = get_posts($args);

            if ($forms) {
                foreach ($forms as $form) {
                    return $form->ID;
                }
            }
        }

        return -1;
    }

    protected function chakavak_get_all_pages()
    {
        $args = array(
            'post_type' => 'page',
            'posts_per_page' => -1
        );
        $catlist = [];
        if ($categories = get_posts($args)) {
            foreach ($categories as $category) {
                (int)$catlist[$category->ID] = $category->post_title;
            }
        } else {
            (int)$catlist['0'] = esc_html__('No Pages Found!', 'chakavak');
        }

        return $catlist;
    }


    protected function register_controls()
    {
        $this->register_general_content_controls();
        $this->register_item_style_controls();
        $this->register_input_style_controls();
        $this->register_general_style_controls();
        $this->register_radio_checkbox_style_controls();
        $this->register_button_controls();
        $this->register_message_content_controls();
        $this->register_other_input_controls();
        $this->register_typography_controls();
    }

    protected function register_general_content_controls()
    {
        $this->start_controls_section(
            'cf7_content',
            [
                'label' => esc_html__('General', 'chakavak'),
            ]
        );

        $this->add_control(
            'cf7_select_form',
            [
                'label' => esc_html__('Select Form', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => $this->get_cf7_forms(),
                'default' => '0'
            ]
        );

        $this->add_control(
            'cf7_redirect_page',
            [
                'label' => esc_html__('On Success Redirect To', 'chakavak'),
                'description' => esc_html__('Select a page which you want users to redirect to when the contact fom is submitted and is successful. Leave Blank to Disable', 'chakavak'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => false,
                'options' => $this->chakavak_get_all_pages(),
            ]
        );

        $this->end_controls_section();

    }

    protected function register_item_style_controls()
    {
        $this->start_controls_section(
            'form_style',
            [
                'label' => esc_html__('Form body style', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Background::get_type(),
            [
                'name' => 'item_bg_color',
                'label' => esc_html__('Background', 'chakavak'),
                'types' => ['classic', 'gradient'],
                'exclude' => ['image'],
                'selector' => '{{WRAPPER}} .chk-cf7__container',

            ]
        );

        $this->add_responsive_control(
            'item_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px'],
                'selectors' => [
                    '{{WRAPPER}} .chk-cf7__container' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'item_border_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-cf7__container' => 'border-radius:{{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'item_box_shadow',
                'selector' => '{{WRAPPER}} .chk-cf7__container',
            ]
        );

        $this->end_controls_section();
    }

    protected function register_general_style_controls()
    {

        $this->start_controls_section(
            'general_style',
            [
                'label' => esc_html__('General style', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'cf7_label_heading',
            [
                'label' => esc_html__('Label', 'chakavak'),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_control(
            'cf7_label_color',
            [
                'label' => esc_html__('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form > p > label ' => 'color:{{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form .form-field > label' => 'color:{{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_input_text_heading',
            [
                'label' => esc_html__('Input text / Textarea / Select', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'cf7_input_padding',
            [
                'label' => __('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_input_radius',
            [
                'label' => __('Border Radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_input_paceholder',
            [
                'label' => esc_html__('Placeholder Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control::placeholder' => 'color:{{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control::-moz-placeholder' => 'color:{{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control::-webkit-input-placeholder' => 'color:{{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_input_color',
            [
                'label' => esc_html__('Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio)' => 'color:{{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_input_border_style',
            [
                'label' => esc_html__('Border Style', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'solid',
                'label_block' => false,
                'options' => chakavak_border_style(),
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio)' => 'border-style: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_input_border_width',
            [
                'label' => __('Border Width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 10,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio)' => 'border-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('cf7_input_tabs');
        $this->start_controls_tab(
            'cf7_input_tab_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );
        $this->add_control(
            'cf7_input_bgcolor_normal',
            [
                'label' => esc_html__('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio)' => 'background-color:{{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'cf7_input_border_color_normal',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'cf7_input_border_width!' => '0',
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio)' => 'border-color:{{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'cf7_input_shadow_normal',
                'selector' => '{{WRAPPER}} .wpcf7-form-control:not(.wpcf7-radio,.wpcf7-checkbox)',
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'cf7_input_tab_focus',
            [
                'label' => __('Focus', 'chakavak'),
            ]
        );
        $this->add_control(
            'cf7_input_bgcolor_focus',
            [
                'label' => esc_html__('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio):focus' => 'background-color:{{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'cf7_input_border_color_focus',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'cf7_input_border_width!' => '0',
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-checkbox,.wpcf7-radio):focus' => 'border-color:{{VALUE}};',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'cf7_input_shadow_focus',
                'fields_options' => [
                    'box_shadow_type' =>
                        [
                            'default' => 'yes'
                        ],
                    'box_shadow' => [
                        'default' =>
                            [
                                'horizontal' => 0,
                                'vertical' => 0,
                                'blur' => 20,
                                'spread' => 0,
                                'color' => 'rgba(0,0,0,0.05)'
                            ]
                    ]
                ],
                'selector' => '{{WRAPPER}} .wpcf7-form-control-wrap .wpcf7-form-control:not(.wpcf7-radio,.wpcf7-checkbox):focus',
            ]
        );
        $this->end_controls_tabs();
        $this->end_controls_section();
    }

    protected function register_input_style_controls()
    {
        $this->start_controls_section(
            'cf7_input_spacing',
            [
                'label' => esc_html__('Spacing & Height', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'cf7_input_margin_top',
            [
                'label' => esc_html__('Between Label & Input', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 60,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input:not([type=submit]):not([type=checkbox]):not([type=radio]),
					 {{WRAPPER}} .wpcf7-form label > .wpcf7-form-control-wrap' => 'margin-top: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_input_margin_bottom',
            [
                'label' => esc_html__('Between Fields', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => -10,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 10,
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input:not([type=submit]):not([type=checkbox]):not([type=radio]),
					 {{WRAPPER}} .wpcf7-form label > .wpcf7-form-control-wrap' => 'margin-bottom: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_textarea_height',
            [
                'label' => esc_html__('Textarea Height', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 500,
                    ],
                ],
                'default' => [
                    'unit' => 'px',
                    'size' => 150,
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form textarea' => 'height:{{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->end_controls_section();
    }

    protected function register_radio_checkbox_style_controls()
    {
        $this->start_controls_section(
            'section_style',
            [
                'label' => esc_html__('Radio / Checkbox', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'cf7_radio_check_body_heading',
            [
                'label' => __('Body', 'chakavak'),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_responsive_control(
            'cf7_radio_check_body_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-checkbox)' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-radio)' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_radio_check_body_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-checkbox)' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-radio)' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_radio_check_body_bg',
            [
                'label' => esc_html__('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-checkbox)' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-radio)' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'cf7_radio_check_body_border',
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'color' => [
                        'default' => 'var(--chk-c-gray-200)',
                    ],
                    'width' => [
                        'default' => [
                            'top' => '1',
                            'right' => '1',
                            'bottom' => '1',
                            'left' => '1',
                            'isLinked' => true,
                        ]
                    ],
                ],
                'selector' => '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-checkbox) , {{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-radio)',
            ]
        );

        $this->add_control(
            'cf7_radio_check_direction',
            [
                'label' => esc_html__('Direction', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'block',
                'options' => [
                    'inline' => esc_html__('Inline', 'chakavak'),
                    'block' => esc_html__('Block', 'chakavak')
                ],
                'prefix_class' => 'chk-radio-check-',
            ]
        );

        $this->add_responsive_control(
            'cf7_radio_check_gap',
            [
                'label' => __('Space Between', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox ' => 'gap: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-radio ' => 'gap: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_radio_check_input_heading',
            [
                'label' => __('Radio button & Checkbox', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'cf7_radio_check_size',
            [
                'label' => _x('Size input', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', 'em', 'rem'],
                'range' => [
                    'px' => [
                        'min' => 10,
                        'max' => 50,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-radio input[type="radio"]' => 'width: {{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}};line-height:{{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .wpcf7-checkbox input[type="checkbox"]' => 'width: {{SIZE}}{{UNIT}}; height:{{SIZE}}{{UNIT}};line-height:{{SIZE}}{{UNIT}}',
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]:checked:before' => 'font-size: calc({{SIZE}}{{UNIT}} *.7);',
                ],
            ]
        );

        $this->add_control(
            'cf7_check_border_radius',
            [
                'label' => esc_html__('Checkbox border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_radio_check_border_width',
            [
                'label' => _x('Border width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 10,
                        'step' => 1,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]' => 'border-width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-radio input[type="radio"]' => 'border-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('cf7_radio_check_tabs');

        $this->start_controls_tab(
            'cf7_radio_check_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'cf7_radio_check_label_color',
            [
                'label' => esc_html__('Label color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox .wpcf7-list-item-label' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio .wpcf7-list-item-label' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_radio_check_border_color',
            [
                'label' => esc_html__('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio input[type=radio]' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_radio_check_bg_color',
            [
                'label' => esc_html__('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio input[type=radio]' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio input[type=radio]:checked:before' => 'box-shadow: {{VALUE}} 0 0 0 2px inset;',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'cf7_radio_check_focus',
            [
                'label' => __('Focus', 'chakavak'),
            ]
        );

        $this->add_control(
            'cf7_radio_check_fucos',
            [
                'label' => esc_html__('color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]:checked' => 'border-color: {{VALUE}};background-color:{{VALUE}};',
                    '{{WRAPPER}} .wpcf7-checkbox input[type=checkbox]:checked+span.wpcf7-list-item-label' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio input[type=radio]:checked' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio input[type=radio]:checked+span.wpcf7-list-item-label' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-radio input[type=radio]:checked:before' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();
    }

    protected function register_button_controls()
    {
        $this->start_controls_section(
            'cf7_submit_button',
            [
                'label' => esc_html__('Submit Button', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'cf7_button_align',
            [
                'label' => esc_html__('Button Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => esc_html__('Left', 'chakavak'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'chakavak'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__('Right', 'chakavak'),
                        'icon' => 'eicon-text-align-right',
                    ],
                    'justify' => [
                        'title' => esc_html__('Justified', 'chakavak'),
                        'icon' => 'eicon-text-align-justify',
                    ],
                ],
                'default' => 'center',
                'prefix_class' => 'chk%s-cf7-button-',
                'toggle' => false,
            ]
        );

        $this->add_responsive_control(
            'cf7_button_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'btn_border_radius',
            [
                'label' => esc_html__('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'cf7_button_border',
                'label' => esc_html__('Border', 'chakavak'),
                'selector' => '{{WRAPPER}} .wpcf7-form input[type="submit"] , .wpcf7-form button[type="submit"]',
            ]
        );

        $this->add_responsive_control(
            'cf7_button_margin',
            [
                'label' => esc_html__('Margin', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'after',
            ]
        );

        $this->start_controls_tabs('tabs_button_style');
        $this->start_controls_tab(
            'tab_button_normal',
            [
                'label' => esc_html__('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_text_color_normal',
            [
                'label' => esc_html__('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_text_bgcolor_normal',
            [
                'label' => esc_html__('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow_normal',
                'selector' => '{{WRAPPER}} .wpcf7-form input[type="submit"] , .wpcf7-form button[type="submit"]',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_button_hover',
            [
                'label' => esc_html__('Hover', 'chakavak'),
            ]
        );

        $this->add_control(
            'button_text_color_hover',
            [
                'label' => esc_html__('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]:is(:hover,:focus)' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]:is(:hover,:focus)' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_text_bgcolor_hover',
            [
                'label' => esc_html__('Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]:is(:hover,:focus)' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]:is(:hover,:focus)' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'button_border_color_hover',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form input[type="submit"]:is(:hover,:focus)' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form button[type="submit"]:is(:hover,:focus)' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'button_box_shadow_hover',
                'selector' => '{{WRAPPER}} .wpcf7-form input[type="submit"]:is(:hover,:focus) , .wpcf7-form button[type="submit"]:is(:hover,:focus)',
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();
        $this->end_controls_section();

    }

    protected function register_message_content_controls()
    {
        $this->start_controls_section(
            'cf7_error_field',
            [
                'label' => esc_html__('Success / Error Message', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'cf7_field_message',
            [
                'label' => esc_html__('Field Validation', 'chakavak'),
                'type' => Controls_Manager::HEADING,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cf7_ot_valid_typography',
                'selector' => '{{WRAPPER}} .wpcf7-form span.wpcf7-not-valid-tip',
            ]
        );

        $this->add_control(
            'cf7_ot_valid_color',
            [
                'label' => esc_html__('Message Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form span.wpcf7-not-valid-tip' => 'color: {{VALUE}};',
                ],
            ]
        );


        $this->add_responsive_control(
            'cf7_ot_valid_margin',
            [
                'label' => esc_html__('Massage margin', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form span.wpcf7-not-valid-tip' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_validation_message',
            [
                'label' => esc_html__('Form Success / Error Validation', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'cf7_validation_message_typography',
                'selector' => '
				{{WRAPPER}} .wpcf7 .wpcf7-validation-errors ,
				{{WRAPPER}} div.wpcf7-mail-sent-ng ,
				{{WRAPPER}} .wpcf7 .wpcf7-acceptance-missing ,
				{{WRAPPER}} .wpcf7-mail-sent-ok ,
				{{WRAPPER}} .wpcf7.sent .wpcf7-response-output',
            ]
        );

        $this->add_responsive_control(
            'cf7_valid_border_size',
            [
                'label' => esc_html__('Border Size', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing ' => 'border-width: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}; border-style: solid;',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_valid_message_radius',
            [
                'label' => esc_html__('Border Radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_valid_message_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'cf7_valid_message_box_margin',
            [
                'label' => esc_html__('Margin', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
                'separator' => 'after',
            ]
        );

        $this->start_controls_tabs('cf7_validation_tabs_style');

        $this->start_controls_tab(
            'cf7_validation_tab_success',
            [
                'label' => esc_html__('Success Message', 'chakavak'),
            ]
        );

        $this->add_control(
            'cf7_success_message_color',
            [
                'label' => esc_html__('Message Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .sent .wpcf7-response-output,
					 {{WRAPPER}} .wpcf7-mail-sent-ok' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_success_message_bgcolor',
            [
                'label' => esc_html__('Message Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .sent .wpcf7-response-output,
					 {{WRAPPER}} .wpcf7-mail-sent-ok' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_success_border_color',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'condition' => [
                    'cf7_valid_border_size!' => '',
                ],
                'selectors' => [
                    '{{WRAPPER}} .sent .wpcf7-response-output,
					 {{WRAPPER}} .wpcf7-mail-sent-ok' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'cf7_validation_tab_error',
            [
                'label' => esc_html__('Error Message', 'chakavak'),
            ]
        );

        $this->add_control(
            'cf7_error_message_color',
            [
                'label' => esc_html__('Message Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .invalid .wpcf7-response-output,
					 {{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_error_message_bgcolor',
            [
                'label' => esc_html__('Message Background Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .invalid .wpcf7-response-output,
					 {{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_error_border_color',
            [
                'label' => esc_html__('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'condition' => [
                    'cf7_valid_border_size!' => '',
                ],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7 .wpcf7-validation-errors,
					 {{WRAPPER}} div.wpcf7-mail-sent-ng,
					 {{WRAPPER}} .wpcf7-acceptance-missing' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    protected function register_other_input_controls()
    {
        $this->start_controls_section(
            'cf7_other_field',
            [
                'label' => esc_html__('Other input fileds', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'cf7_other_field_body_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-file)' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-acceptance)' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_other_field_body_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-file)' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-acceptance)' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'cf7_other_field_body_bg',
            [
                'label' => esc_html__('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-file)' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-acceptance)' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name' => 'cf7_other_field_body_border',
                'fields_options' => [
                    'border' => [
                        'default' => 'solid',
                    ],
                    'color' => [
                        'default' => 'var(--chk-c-gray-200)',
                    ],
                    'width' => [
                        'default' => [
                            'top' => '1',
                            'right' => '1',
                            'bottom' => '1',
                            'left' => '1',
                            'isLinked' => true,
                        ]
                    ],
                ],
                'selector' => '{{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-file) , {{WRAPPER}} .wpcf7-form-control-wrap:has(.wpcf7-acceptance)',
            ]
        );

        $this->end_controls_section();
    }

    protected function register_typography_controls()
    {
        $this->start_controls_section(
            'cf7_typo',
            [
                'label' => esc_html__('Typography', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'cf7_label_typo',
            [
                'label' => esc_html__('Form Label', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'form_label_typography',
                'selector' => '{{WRAPPER}} .wpcf7 .wpcf7-form>p>label ,{{WRAPPER}} .wpcf7-form .form-field > label',
            ]
        );


        $this->add_control(
            'cf7_input_typo',
            [
                'label' => esc_html__('Form input', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'input_typography',
                'selector' => '
				    {{WRAPPER}} .wpcf7-form input:not([type=submit],[type="checkbox"],[type="radio"]),
				    {{WRAPPER}} .wpcf7-form input:not([type=submit]),
					{{WRAPPER}} .wpcf7-form input::placeholder,
					{{WRAPPER}} .wpcf7-form select,
					{{WRAPPER}} .wpcf7-form textarea,
					{{WRAPPER}} .wpcf7-form textarea::placeholder,
					{{WRAPPER}} .wpcf7-form input[type=range]::-webkit-slider-thumb',
            ]
        );

        $this->add_control(
            'cf7_label_radio_typo',
            [
                'label' => esc_html__('Form Radio button & checkbox', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );
        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'radio_checkbox_typography',
                'selector' => '
				    {{WRAPPER}} .wpcf7-form .wpcf7-acceptance>.wpcf7-list-item span.wpcf7-list-item-label,
                    {{WRAPPER}} .wpcf7-form .wpcf7-checkbox .wpcf7-list-item-label,
                    {{WRAPPER}} .wpcf7-form .wpcf7-radio .wpcf7-list-item-label',
            ]
        );

        $this->add_control(
            'btn_typography_label',
            [
                'label' => esc_html__('Button', 'chakavak'),
                'type' => Controls_Manager::HEADING,
                'separator' => 'before',
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'btn_typography',
                'label' => esc_html__('Typography', 'chakavak'),
                'selector' => '
				{{WRAPPER}} .wpcf7-form input[type=submit],
				{{WRAPPER}} .wpcf7-form button[type=submit]',
            ]
        );

        $this->end_controls_section();
    }

    protected function render_editor_script()
    {
        if (\Elementor\Plugin::instance()->editor->is_edit_mode() === false) {
            return;
        }

        $pre_url = wpcf7_get_request_uri();

        if (strpos($pre_url, 'admin-ajax.php') === false) {
            return;
        }

        ?>
        <script type="text/javascript">
            jQuery(document).ready(function ($) {

                $('.chk-cf7__container').each(function () {
                    var $node_id = '<?php echo $this->get_id(); ?>';
                    var scope = $('[data-id="' + $node_id + '"]');
                    var selector = $(this);

                    if (selector.closest(scope).length < 1) {
                        return;
                    }

                    if (selector.find('div.wpcf7 > form').length < 1) {
                        return;
                    }

                    selector.find('div.wpcf7 > form').each(function () {
                        var $form = $(this);
                        wpcf7.initForm($form);
                    });
                });
            });
        </script>
        <?php
    }

    protected function render()
    {
        if (!class_exists('WPCF7_ContactForm')) {
            return;
        }
        static $v_veriable = 0;
        $settings = $this->get_settings_for_display();
        $node_id = $this->get_id();
        $field_options = array();
        $classname = '';

        $args = array(
            'post_type' => 'wpcf7_contact_form',
            'posts_per_page' => -1,
        );

        $forms = get_posts($args);
        $field_options['0'] = 'select';
        if ($forms) {
            foreach ($forms as $form) {
                $field_options[$form->ID] = $form->post_title;
            }
        }

        //$this->add_inline_editing_attributes( 'chk_cf7_title' );
        $forms = $this->get_cf7_forms();

        $html = '';
        if (!empty($forms) && !isset($forms[-1])) {
            if (0 == $settings['cf7_select_form']) {
                $html = esc_html__('Please select a Contact Form 7 from dropdown.', 'chakavak');
            } else {
                ?>
                <div class="chk-cf7__container">
                    <div class="chk-cf7 elementor-shortcode elementor-clickable chk-cf7-<?php echo $v_veriable; ?>>">
                        <?php
                        if ($settings['cf7_select_form']) {
                            echo do_shortcode('[contact-form-7 id=' . $settings['cf7_select_form'] . ']');
                        }
                        ?>
                    </div>
                </div>
                <?php
            }
        } else {
            $html = esc_html__('Contact Form not found. Please add contact form from Dashboard > Contact > Add New.', 'chakavak');
        }
        echo $html;

        if (!empty($settings['cf7_redirect_page'])) { ?>
            <script>
                var theform = document.querySelector('.chk-cf7-<?php echo $v_veriable; ?>');
                theform.addEventListener('wpcf7mailsent', function (event) {
                    location = '<?php echo get_permalink($settings['cf7_redirect_page']); ?>';
                }, false);
            </script>

            <?php $v_veriable++;
        }
        $this->render_editor_script();
    }

    protected function content_template()
    {
    }
}

Plugin::instance()->widgets_manager->register(new chakavak_CF7());
