<?php

namespace Elementor;
defined('ABSPATH') or exit;

/**
 * Widget Name: Button
 */
class Chakavak_button extends Widget_Base
{
    const NAME = 'chk_button';
    const CATEGORY = 'chakavak_widgets';

    public function get_name()
    {
        return self::NAME;
    }

    public function get_categories()
    {
        return array(self::CATEGORY);
    }

    public function get_title()
    {
        return __('Button', 'chakavak');
    }

    public function get_icon()
    {
        return 'chk-el-icon eicon-button';
    }

    public function get_keywords()
    {
        return ['button', esc_html__('Button', 'chakavak')];
    }

    protected function register_controls()
    {
        $this->start_controls_section(
            'btn_settings',
            [
                'label' => esc_html__('Button', 'chakavak'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'btn_type',
            [
                'label' => esc_html__('Button Type', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'flat',
                'options' => [
                    'flat' => esc_html__('Flat Button', 'chakavak'),
                    'outline' => esc_html__('Outline Button', 'chakavak'),
                ],
            ]
        );

        $this->add_control(
            'btn_size',
            [
                'label' => __('Button Size', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'chk-button__default' => esc_html__('Size default', 'chakavak'),
                    'chk-button__small' => esc_html__('Size small', 'chakavak'),
                    'chk-button__large' => esc_html__('Size large', 'chakavak')
                ],
                'default' => 'chk-button__default',
            ]
        );

        $this->add_control(
            'label',
            [
                'label' => __('Label', 'chakavak'),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 3,
                'dynamic' => [
                    'active' => true,
                ],
                'label_block' => true,
                'default' => __('Click here', 'chakavak'),
                'placeholder' => __('Enter label button', 'chakavak'),
            ]
        );

        $this->add_control(
            'link',
            [
                'label' => __('Link', 'chakavak'),
                'type' => Controls_Manager::URL,
                'dynamic' => [
                    'active' => true,
                ],
                'placeholder' => 'https://your-link.com',
                'default' => [
                    'url' => '#',
                ],
            ]
        );

        $this->add_control(
            'btn_css_id',
            [
                'label' => __('Button ID', 'chakavak'),
                'type' => Controls_Manager::TEXT,
                'dynamic' => [
                    'active' => true,
                ],
                'default' => '',
                'label_block' => true,
                'title' => __('Add your custom id WITHOUT the Pound key. e.g: my-id', 'chakavak'),
                'description' => __('Please make sure the ID is unique and not used elsewhere on the page this form is displayed. This field allows <code>A-z 0-9</code> & underscore chars without spaces.', 'chakavak'),
                'separator' => 'before',

            ]
        );

        $this->add_responsive_control(
            'align',
            [
                'label' => __('Alignment', 'chakavak'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'left' => [
                        'title' => __('Left', 'chakavak'),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => __('Center', 'chakavak'),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => __('Right', 'chakavak'),
                        'icon' => 'eicon-text-align-right',
                    ],
                ],
                'prefix_class' => 'elementor%s-align-',
                'selectors' => [
                    '{{WRAPPER}}' => 'text-align: {{VALUE}};',
                ],
                'default' => '',
            ]
        );

        $this->add_control(
            'selected_icon',
            [
                'label' => esc_html__('Icon', 'chakavak'),
                'type' => Controls_Manager::ICONS,
                'fa4compatibility' => 'icon',
                'skin' => 'inline',
                'label_block' => false,
                'default' => [
                    'value' => 'ti-phone-call',
                    'library' => 'theme-icons',
                ],
            ]
        );

        $this->add_control(
            'icon_align',
            [
                'label' => esc_html__('Icon Position', 'chakavak'),
                'type' => Controls_Manager::SELECT,
                'default' => 'before',
                'options' => [
                    'before' => esc_html__('Before', 'chakavak'),
                    'after' => esc_html__('After', 'chakavak'),
                ],
                'condition' => [
                    'selected_icon[value]!' => ''
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'style_section',
            [
                'label' => __('General', 'chakavak'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'width',
            [
                'label' => esc_html__('Width', 'chakavak'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%', 'vw'],
                'range' => [
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .chk-button' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'btn_padding',
            [
                'label' => esc_html__('Padding', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'rem', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'btn_radius',
            [
                'label' => __('Border radius', 'chakavak'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .chk-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'btn_typography',
                'selector' => '{{WRAPPER}} .chk-button',
            ]
        );

        $this->start_controls_tabs('tabs_button_style');

        $this->start_controls_tab(
            'tab_button_normal',
            [
                'label' => __('Normal', 'chakavak'),
            ]
        );

        $this->add_control(
            'btn_text_color',
            [
                'label' => __('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-button' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'btn_background_color',
            [
                'label' => __('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-button' => 'background-color: {{VALUE}};',
                ],
                'condition' => [
                    'btn_type' => ['flat'],
                ],
            ]
        );

        $this->add_control(
            'btn_border_color',
            [
                'label' => __('Border color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-button.chk-button__flat' => 'border-color: {{VALUE}};',
                    '{{WRAPPER}} .chk-button.chk-button__outline' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'btn_box_shadow',
                'selector' => '{{WRAPPER}} .chk-button',
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'tab_button_hover',
            [
                'label' => __('Hover', 'chakavak'),
            ]
        );

        $this->add_control(
            'hover_color',
            [
                'label' => __('Text Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-button:is(:hover,:focus)' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'btn_background_color_hover',
            [
                'label' => __('Background color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'default' => '',
                'selectors' => [
                    '{{WRAPPER}} .chk-button:is(:hover,:focus)' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'border_hover_color',
            [
                'label' => __('Border Color', 'chakavak'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .chk-button:is(:hover,:focus)' => 'border-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'btn_box_shadow_hover',
                'selector' => '{{WRAPPER}} .chk-button:is(:hover,:focus)',
                'fields_options' => [
                    'box_shadow_type' =>
                        [
                            'default' => 'yes'
                        ],
                    'box_shadow' => [
                        'default' =>
                            [
                                'horizontal' => 0,
                                'vertical' => 8,
                                'blur' => 15,
                                'spread' => 0,
                                'color' => 'rgba(0,0,0,0.1)'
                            ]
                    ]
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $this->add_render_attribute('button-wrapper', 'class', 'chk-button__wrapper');

        if (!empty($settings['link']['url'])) {
            $this->add_render_attribute('button', 'href', $settings['link']['url']);
            if ($settings['link']['is_external']) {
                $this->add_render_attribute('button', 'target', '_blank');
            }
            if ($settings['link']['nofollow']) {
                $this->add_render_attribute('button', 'rel', 'nofollow');
            }
        }

        if (!empty($settings['btn_css_id'])) {
            $this->add_render_attribute('button', 'id', $settings['btn_css_id']);
        }

        $this->add_render_attribute('button', 'class', 'chk-button');
        $this->add_render_attribute('button', 'class', 'chk-button__' . $settings['btn_type'] . '');
        $this->add_render_attribute('button', 'class', $settings['btn_size']);

        if (!empty($settings['icon_align'])) {
            $this->add_render_attribute('button', 'class', 'chk-button__icon-' . $settings['icon_align']);
        }

        $migrated = isset($settings['__fa4_migrated']['selected_icon']);
        $is_new = empty($settings['icon']) && Icons_Manager::is_migration_allowed();

        ?>
        <div class="<?php echo $this->get_name(); ?>-element chk-element">
            <div <?php echo $this->get_render_attribute_string('button-wrapper'); ?>>
                <a <?php echo $this->get_render_attribute_string('button'); ?>>
			        <span>
			            <?php if (!empty($settings['icon']) || !empty($settings['selected_icon']['value'])) : ?>
                            <?php if ($is_new || $migrated) :
                                Icons_Manager::render_icon($settings['selected_icon'], ['aria-hidden' => 'true']);
                            else : ?>
                                <i class="<?php echo esc_attr($settings['icon']); ?>" aria-hidden="true"></i>
                            <?php endif; ?>
                        <?php endif; ?>
                        <?php $this->print_unescaped_setting('label'); ?>
			        </span>
                </a>
            </div>
        </div>
        <?php
    }

    protected function content_template()
    {
    }
}

Plugin::instance()->widgets_manager->register(new Chakavak_button());